import { ICommand, ICommandResponse } from './ICommand';
import { cache } from '../../shared/@decorators/function';
import { AStar } from '../pathfinding/astar';
import DealsDb from '../resources/deals';
import { IDeal } from '../../shared/models/deals';
import { IDirection } from '../../shared/models/directions';

interface IDirectionResponse extends ICommandResponse, IDirection { }

/**
 * Command to manage pathfinding
 */
export class Direction implements ICommand<IDirectionResponse> {
    private pathfindingAlgo: AStar<IDeal, string>;
    public constructor() {
        this.pathfindingAlgo = new AStar();
        this.pathfindingAlgo
            .addHeuristics('duration', (node, previousScore) => {
                return parseInt(node.duration.h, 10) * 60 + parseInt(node.duration.m, 10) + previousScore;
            })
            .addHeuristics('cost', (node, previousScore) => {
                return node.cost * ((100 - node.discount) / 100) + previousScore;
            })
            .setNodeService(nodeValue => DealsDb.find(nodeValue))
            .setStepResolver(node => node.arrival.toLowerCase());
    }

    /**
     * Find the perfect path according to given method
     * @param cityFrom Name of departure city
     * @param cityTo Name of the arrival city
     * @param method Scoring method: 'duration' or 'cost'
     */
    @cache(10)
    public execute(cityFrom: string = '', cityTo: string = '', method: string = '') {
        return new Promise<IDirectionResponse>((resolve, reject) => {
            Promise.all([
                this.pathfindingAlgo.find(cityFrom.toLowerCase(), cityTo.toLowerCase(), method),
                DealsDb.getCurrency()
            ]).then(response => {
                const [cluster, currency] = response;
                const steps = !cluster ? [] : cluster.getPath();
                const min = steps.reduce((prev, current) => prev + parseInt(current.duration.m, 10) + parseInt(current.duration.h, 10) * 60, 0);
                resolve({
                    steps,
                    total: {
                        cost: steps.reduce((prev, current) => prev + (current.cost * (100 - current.discount) / 100), 0),
                        duration: {
                            h: `0${Math.floor(min / 60).toString()}`.substr(-2),
                            m: `0${(min % 60).toString()}`.substr(-2)
                        }
                    },
                    currency
                });
            }).catch(reject);
        });
    }
}
