interface ICacheItem {
    /**
     * Creation date
     */
    date: number;
    /**
     * Response content value
     */
    response: any;
    /**
     * Unique hash id
     */
    hash: string;
}

/**
 * Cache response from function which are called many time with same parameters.
 * This implementation works only with primitive parameters.
 * @param cacheItemSize Number of item to cache
 */
export function cache(cacheItemSize: number) {
    // Basic cache. Can be improved by using call number.
    const cacheMap = new Map<string, ICacheItem>();

    return function (target: any, method: string, descriptor: PropertyDescriptor) {
        const originalFunction = target[method];
        descriptor.value = target[method] = function (...args: any[]) {
            const hash = JSON.stringify(args); // Only primitive for the moment.

            if (cacheMap.has(hash)) {
                return cacheMap.get(hash)!.response;
            }

            const response = originalFunction.apply(this, args);
            const date = Date.now();
            cacheMap.set(hash, {
                date,
                response,
                hash
            });

            if (cacheMap.size >= cacheItemSize) {
                cleanCache(cacheMap, cacheItemSize);
            }

            if (response instanceof Promise) {
                response.catch(_ => cacheMap.delete(hash));
            }
            return response;
        };
    };
}

function cleanCache(map: Map<string, ICacheItem>, maxItems: number) {
    const result: Array<ICacheItem> = [];
    map.forEach((value, key) => result.push(value));

    const orderedList = result.sort((a, b) => a.date - b.date);
    while (orderedList.length > maxItems) {
        map.delete(orderedList.shift()!.hash);
    }
}

